package furny.ga;

import furny.entities.Furniture;
import furny.furndb.FurnCache;
import furny.ga.util.Distance;
import furny.ga.util.FurnLayoutIOUtil;
import ga.core.GA;
import ga.core.algorithm.util.ClusterUtil;
import ga.core.algorithm.util.RandomSingleton;
import ga.core.individual.IClusterableIndividual;
import ga.core.individual.ICostInfo;
import ga.core.individual.IDebugInfo;
import ga.core.individual.IIntervalFitness;
import ga.core.validation.GAContext;
import ga.view.interfaces.IPhenotypeSpace;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Individual for furniture layouts. This supports clustering, interval fitness,
 * cost info output and debug infos.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class FurnLayoutIndividual implements
    IClusterableIndividual<FurnLayoutIndividual>, IIntervalFitness, IDebugInfo,
    ICostInfo {
  private static Random rnd = RandomSingleton.getRandom();

  /**
   * Generator for unique ids.
   */
  private static final AtomicLong ID_GENERATOR = new AtomicLong();

  private final long id;

  /**
   * List of furniture entries. This is the genotype of the individual.
   */
  private final FurnEntryList furnitures = new FurnEntryList();
  private double fitnessCenter = UNEVALUATED;
  private double fitnessMin;
  private double fitnessMax;

  /**
   * The GA context is always required.
   */
  private GAContext context;

  /**
   * Creates a new individual using the given context. It will have a unique id.
   * 
   * @param context
   *          The GA context.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public FurnLayoutIndividual(final GAContext context) {
    this();
    setContext(context);
  }

  /**
   * Creates a new individual. It will have a unique id. The GA context must be
   * set later.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public FurnLayoutIndividual() {
    id = ID_GENERATOR.incrementAndGet();
  }

  @Override
  public void setContext(final GAContext context) {
    this.context = context;
  }

  @Override
  public GAContext getContext() {
    return context;
  }

  @Override
  public long getId() {
    return id;
  }

  @Override
  public void setFitness(final double fitness) {
    setFitnessInterval(fitness, 0d);
  }

  @Override
  public double getFitness() {
    return fitnessCenter;
  }

  @Override
  public double getMaxFitness() {
    return fitnessMax;
  }

  @Override
  public double getMinFitness() {
    return fitnessMin;
  }

  @Override
  public double getFitnessWidth() {
    return fitnessMax - fitnessMin;
  }

  @Override
  public void setFitnessInterval(final double center, final double width) {
    this.fitnessCenter = center;
    fitnessMin = center - (width / 2d);
    fitnessMax = fitnessMin + width;
  }

  @Override
  public void setFitnessLimits(final double min, final double max) {
    this.fitnessMin = min;
    this.fitnessMax = max;
    this.fitnessCenter = (max + min) / 2d;
  }

  @Override
  public void initRandomly() {
    fitnessCenter = UNEVALUATED;
    fitnessMin = 0d;
    fitnessMax = 0d;
    furnitures.clear();

    // fallback values
    float minX = -10f;
    float maxX = 10f;
    float minY = -10f;
    float maxY = 10f;

    IPhenotypeSpace space = null;

    if (context != null) {
      final Object o = context.get(GA.KEY_VALIDATION_SPACE);

      if (o != null && o instanceof IPhenotypeSpace) {
        space = (IPhenotypeSpace) o;

        minX = (float) space.getOutterBounds().getX();
        maxX = (float) (space.getOutterBounds().getX() + space
            .getOutterBounds().getWidth());

        minY = (float) space.getOutterBounds().getY();
        maxY = (float) (space.getOutterBounds().getY() + space
            .getOutterBounds().getHeight());
      }
    }

    int genomeLength;

    // fallbackvalues
    int genomeMinLength = 1;
    int genomeMaxLength = 6;

    if (context != null) {
      if (context.containsKey(GA.KEY_GENOME_MIN_LENGTH)) {
        genomeMinLength = (Integer) context.get(GA.KEY_GENOME_MIN_LENGTH);
      }

      if (context.containsKey(GA.KEY_GENOME_MAX_LENGTH)) {
        genomeMaxLength = (Integer) context.get(GA.KEY_GENOME_MAX_LENGTH);
      }
    }

    // add 1 to include the max length because Random.nextInt(n) excludes n
    genomeLength = rnd.nextInt(1 + genomeMaxLength - genomeMinLength)
        + genomeMinLength;

    final List<Furniture> all = FurnCache.getInstance().getAllFurnitures();

    for (int i = 0; i < genomeLength; i++) {
      final int x = Math
          .round(((rnd.nextFloat() * (maxX - minX)) + minX) * 100);
      final int y = Math
          .round(((rnd.nextFloat() * (maxY - minY)) + minY) * 100);
      final int rotationSteps = rnd.nextInt(4);

      final Furniture f = all.get(rnd.nextInt(all.size()));

      furnitures.add(new FurnEntry(new RoomVector(x, y, rotationSteps), f));
    }
  }

  /**
   * Getter for the genotype of the individual.
   * 
   * @return The list of furniture entries.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public FurnEntryList getFurnitures() {
    return furnitures;
  }

  /**
   * Getter for the simple genotype. This is just an array of {@code long}
   * numbers.
   * 
   * @return The simple genotype.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public long[][] getSimpleGenotype() {
    final long[][] store = new long[furnitures.size()][4];

    for (int i = 0; i < furnitures.size(); i++) {
      final FurnEntry entry = furnitures.get(i);
      store[i][0] = entry.getFurniture().getId();
      store[i][1] = entry.getVector().getXGene();
      store[i][2] = entry.getVector().getYGene();
      store[i][3] = entry.getVector().getRotationSteps();
    }

    return store;
  }

  @Override
  public boolean isEvaluated() {
    return fitnessCenter != UNEVALUATED;
  }

  @Override
  public FurnLayoutIndividual clone() {
    final FurnLayoutIndividual newInd = new FurnLayoutIndividual(context);

    for (final FurnEntry entry : furnitures) {
      newInd.furnitures.add(entry.clone());
    }

    return newInd;
  }

  @Override
  public String toString() {
    final StringBuilder sb = new StringBuilder();
    FurnLayoutIOUtil.writeSimpleGenotype(getSimpleGenotype(), sb);
    FurnLayoutIOUtil.writeFitness(this, sb);
    return sb.toString();
  }

  @Override
  public String getDebugString() {
    String fitnessString;

    if (isEvaluated()) {
      if (fitnessMin != fitnessMax) {
        fitnessString = String.format(Locale.ENGLISH, "[%.2f;%.2f]",
            fitnessMin, fitnessMax);
      } else {
        fitnessString = String.format(Locale.ENGLISH, "%.2f", fitnessCenter);
      }
    } else {
      fitnessString = "N/A";
    }

    return id + " " + getGenotypeString() + " Fitness: " + fitnessString;
  }

  @Override
  public String getIdString() {
    return String.valueOf(id);
  }

  @Override
  public double getCosts() {
    double costs = 0d;

    for (final FurnEntry fe : furnitures) {
      costs += fe.getFurniture().getMetaData().getPrice();
    }

    return costs;
  }

  @Override
  public String getCostString() {
    return String.format(Locale.ENGLISH, "%.2f Euro", getCosts());
  }

  @Override
  public String getGenotypeString() {
    final StringBuilder sb = new StringBuilder();
    FurnLayoutIOUtil.writeSimpleGenotypeSet(getSimpleGenotype(), sb);
    return sb.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (obj == null || obj.getClass() != getClass()) {
      return false;
    }

    final FurnLayoutIndividual that = (FurnLayoutIndividual) obj;

    return this.furnitures.containsAll(that.furnitures)
        && that.furnitures.containsAll(this.furnitures);
  }

  @Override
  public int hashCode() {
    Collections.sort(furnitures);
    return furnitures.hashCode();
  }

  @Override
  public FurnLayoutIndividual centroidOf(
      final Collection<FurnLayoutIndividual> c) {
    return ClusterUtil.calculateCentroid(c);
  }

  @Override
  public double distanceFrom(final FurnLayoutIndividual ind2) {
    return Distance.calcDistance2(this, ind2, context);
  }
}
